<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class VisitReimbursement extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;

    public const STATUS_RADIO = [
        'pending'  => 'Pending',
        'approved' => 'Approved',
        'denied'   => 'Denied',
    ];

    public $table = 'visit_reimbursements';

    public $orderable = [
        'id',
        'user.name',
        'task.name',
        'approved_by.name',
        'approved_at',
        'amount',
        'visit.date',
        'status',
    ];

    public $filterable = [
        'id',
        'user.name',
        'task.name',
        'approved_by.name',
        'approved_at',
        'amount',
        'visit.date',
        'status',
    ];

    protected $dates = [
        'approved_at',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $fillable = [
        'user_id',
        'task_id',
        'approved_by_id',
        'approved_at',
        'amount',
        'visit_id',
        'status',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function task()
    {
        return $this->belongsTo(Task::class);
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class);
    }

    public function getApprovedAtAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setApprovedAtAttribute($value)
    {
        if ($value instanceof Carbon || $value instanceof \DateTime)
            $this->attributes['approved_at'] = $value;
        else
            $this->attributes['approved_at'] = $value ? Carbon::createFromFormat(config('project.datetime_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    public function visit()
    {
        return $this->belongsTo(EmployeeAttendanceLog::class);
    }

    public function routes()
    {
        return $this->hasMany(VisitRoute::class, 'reimbursement_id', 'id');
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }


    public function updateAmount()
    {
        $total = $this->routes()->sum('route_cost');
        $this->update([
            'amount' => $total,
        ]);
    }


    public function getStatusLabelAttribute($value)
    {
        return static::STATUS_RADIO[$this->status] ?? null;
    }
}
