<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;

class ProductTaskEventLog extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;

    public const STATUS_SELECT = [
        'pending'  => 'Pending',
        // 'Approved' => 'Approved',
        // 'Denied'   => 'Denied',
        'requested'  => 'Requested',
        'accepted'   => 'Accepted',
    ];

    public $table = 'product_task_event_logs';

    public $orderable = [
        'id',
        'task_event_log.event_type',
        'product.name',
        'quantity',
        'status',
        'approved_by.name',
        'approved_at',
    ];

    public $filterable = [
        'id',
        'task_event_log.event_type',
        'product.name',
        'quantity',
        'status',
        'approved_by.name',
        'approved_at',
    ];

    protected $dates = [
        'approved_at',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    protected $fillable = [
        'task_event_log_id',
        'product_id',
        'quantity',
        'status',
        'approved_by_id',
        'approved_at',
        'description',
        'unlisted',
        'quantity_ordered',
        'is_ordered',
        'info',
        'extras',
        'additional_info',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'unlisted' => 'boolean',
        'is_ordered' => 'boolean',
        'info' => 'array',
        'extras' => 'array',
        'additional_info' => 'array',
    ];

    public function taskEventLog()
    {
        return $this->belongsTo(TaskEventLog::class);
    }

    public function product()
    {
        return $this->belongsTo(Product::class);
    }
    // accessor status
    public function getStatusAttribute($value)
    {
        return $value ? $value : 'pending';
    }


    // accessor status_label
    public function getStatusLabelAttribute($value)
    {
        return static::STATUS_SELECT[$this->status] ?? null;
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class);
    }

    public function getApprovedAtAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setApprovedAtAttribute($value)
    {
        $this->attributes['approved_at'] = $value ? Carbon::createFromFormat(config('project.datetime_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    # product_name
    public function getProductNameAttribute()
    {
        if ($this->unlisted) {
            return Arr::get($this->info, 'name');
        } else {
            return $this->product->name ?? "";
        }
    }
    # product_model
    public function getProductModelAttribute()
    {
        if ($this->unlisted) {
            return Arr::get($this->info, 'model');
        } else {
            return $this->product->model ?? "";
        }
    }

    // hasMany ProoductRequisition
    public function productRequisitions()
    {
        return $this->hasMany(ProductRequisition::class);
    }
}
