<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class ProductRequisition extends Model
{
    use HasFactory;

    public const STATUS_SELECT = [
        'pending'  => 'Pending',
        // 'Approved' => 'Approved',
        // 'Denied'   => 'Denied',
        'requested'  => 'Requested',
        'accepted'   => 'Accepted',
    ];

    protected $fillable = [
        'requested_quantity',
        'sent_quantity',
        'status',
        'product_id',
        'requested_by_id',
        'requested_by_type',
        'requested_to_id',
        'requested_to_type',
        'requested_for_id',
        'requested_for_type',
        'accepted_by_id',
        'accepted_by_type',
        'p_listed',
        'product_info',
        'info',
        'additional',
        'accepted_at',
        'product_task_event_log_id',
        'task_event_log_id',
        'parent_id',
    ];

    protected $casts = [
        'requested_by_id' => 'integer',
        'requested_to_id' => 'integer',
        'requested_for_id' => 'integer',
        'accepted_by_id' => 'integer',
        'product_info' => 'array',
        'info' => 'array',
        'additional' => 'array',
        'accepted_at' => 'datetime',
    ];


    public function product()
    {
        return $this->belongsTo(Product::class);
    }

    public function requestedBy()
    {
        return $this->morphTo();
    }

    public function requestedTo()
    {
        return $this->morphTo();
    }

    public function requestedFor()
    {
        return $this->morphTo();
    }

    public function acceptedBy()
    {
        return $this->morphTo();
    }


    // accessor status
    public function getStatusAttribute($value)
    {
        return $value ? $value : 'pending';
    }


    // accessor status_label
    public function getStatusLabelAttribute($value)
    {
        return static::STATUS_SELECT[$this->status] ?? null;
    }


    public function productTaskEventLog()
    {
        return $this->belongsTo(ProductTaskEventLog::class);
    }

    public function taskEventLog()
    {
        return $this->belongsTo(TaskEventLog::class);
    }

    // accessor p_name
    public function getPNameAttribute()
    {
        if ($this->product) {
            return $this->product->name;
        } else {
            return Arr::get($this->product_info, 'name');
        }
    }
    // accessor p_description

    public function getPDescriptionAttribute()
    {
        if ($this->product) {
            return $this->product->description;
        } else {
            return Arr::get($this->product_info, 'model');
        }
    }

    // belongsTo ProductRequisition
    public function parent()
    {
        return $this->belongsTo(ProductRequisition::class);
    }

    // hasMany ProductRequisition
    public function children()
    {
        return $this->hasMany(ProductRequisition::class, 'parent_id');
    }

    // hasOne ProductRequisition
    public function child()
    {
        return $this->hasOne(ProductRequisition::class, 'parent_id');
    }



    /*
    static function to create a ProductRequisition entity from given
        requestedFor:Circle
        requestedTo:Zone
        requestedBy:User
        productDetails:ProductTaskEventLog
    as parameters
    */

    public static function createRequisitionCircleToZoneFromProductTaskEventLog(Circle $requestedFor, Zone $requestedTo, User $requestedBy, ProductTaskEventLog $productDetails)
    {
        $productRequisition = new ProductRequisition();
        $productRequisition->requested_for_id = $requestedFor->id;
        $productRequisition->requested_for_type = get_class($requestedFor);
        $productRequisition->requested_to_id = $requestedTo->id;
        $productRequisition->requested_to_type = get_class($requestedTo);
        $productRequisition->requested_by_id = $requestedBy->id;
        $productRequisition->requested_by_type = get_class($requestedBy);
        $productRequisition->product_task_event_log_id = $productDetails->id;
        $productRequisition->task_event_log_id = $productDetails->task_event_log_id;
        $productRequisition->product_id = $productDetails->product_id;
        $productRequisition->requested_quantity = $productDetails->quantity;
        $productRequisition->status = 'pending';
        if($productDetails->product){
            $productRequisition->p_listed = true;
            // $productRequisition->product_info = null;
        }
        else {
            $productRequisition->p_listed = false;
            $productRequisition->product_info = $productDetails->info;
            // $productRequisition->product_info = [
            //     'name' => $productDetails->product_name,
            //     'model' => $productDetails->product_model,
            // ];
        }
        $productRequisition->save();
        return $productRequisition;
    }

    // createRequisitionToAdminFromChild
    public static function createRequisitionToAdminFromChild(ProductRequisition $parent, User $requestedBy )
    {
        $productRequisition = new ProductRequisition();
        // requestedFor = $parent->requestedTo
        $productRequisition->requested_for_id = $parent->requested_to_id;
        $productRequisition->requested_for_type = $parent->requested_to_type;
        // requestedTo = Admin
        $productRequisition->requested_to_id = 1;
        $productRequisition->requested_to_type = 'App\Models\Admin';

        $productRequisition->requested_by_id = $requestedBy->id;
        $productRequisition->requested_by_type = get_class($requestedBy);
        $productRequisition->product_task_event_log_id = $parent->product_task_event_log_id;
        $productRequisition->task_event_log_id = $parent->task_event_log_id;
        $productRequisition->product_id = $parent->product_id;
        $productRequisition->requested_quantity = $parent->requested_quantity;
        $productRequisition->status = 'pending';
        $productRequisition->parent_id = $parent->id;
    }

    // acceptRequisition
    public function acceptRequisition(User $acceptedBy)
    {
        $this->accepted_by_id = $acceptedBy->id;
        $this->accepted_by_type = get_class($acceptedBy);
        $this->accepted_at = now();
        $this->status = 'accepted';
        $this->save();

        // if($this->requested_for_type ==  Circle::class) && has productTaskEventLog
        // set the productTaskEventLog status to accepted with updated sent quantity
        // and subtract the product quantity from the circle
        if($this->requested_for_type == Circle::class && $this->product_task_event_log_id){
            $productTaskEventLog = $this->productTaskEventLog;
            $productTaskEventLog->status = 'accepted';
            $productTaskEventLog->sent_quantity = $this->requested_quantity;
            $productTaskEventLog->save();

            $circle = $this->requestedFor;
            $circle->products()->updateExistingPivot($this->product_id, [
                'quantity' => DB::raw('quantity - ' . $this->requested_quantity)
            ]);
        }

    }

}
