<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class Product extends Model implements HasMedia
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;
    use InteractsWithMedia;

    public $table = 'products';

    public $orderable = [
        'id',
        'name',
        'sku',
        'description',
        'price',
        'hsn_code',
    ];

    public $filterable = [
        'id',
        'name',
        'sku',
        'description',
        'price',
        'hsn_code',
        'category.name',
        // 'tag.name',
    ];

    protected $appends = [
        'photo',
    ];

    protected $fillable = [
        'name',
        'sku',
        'description',
        'price',
        'hsn_code',
        'gst_rate',
        'cgst_rate',
        'sgst_rate',
        'igst_rate',
        'taxable_price',
    ];

    protected $dates = [
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    public function registerMediaConversions(Media $media = null): void
    {
        $thumbnailWidth  = 50;
        $thumbnailHeight = 50;

        $thumbnailPreviewWidth  = 120;
        $thumbnailPreviewHeight = 120;

        $this->addMediaConversion('thumbnail')
            ->width($thumbnailWidth)
            ->height($thumbnailHeight)
            ->fit('crop', $thumbnailWidth, $thumbnailHeight);
        $this->addMediaConversion('preview_thumbnail')
            ->width($thumbnailPreviewWidth)
            ->height($thumbnailPreviewHeight)
            ->fit('crop', $thumbnailPreviewWidth, $thumbnailPreviewHeight);
    }

    public function category()
    {
        return $this->belongsToMany(ProductCategory::class);
    }

    public function tag()
    {
        return $this->belongsToMany(ProductTag::class);
    }

    public function getPhotoAttribute()
    {
        return $this->getMedia('product_photo')->map(function ($item) {
            $media = $item->toArray();
            $media['url'] = $item->getUrl();
            $media['thumbnail'] = $item->getUrl('thumbnail');
            $media['preview_thumbnail'] = $item->getUrl('preview_thumbnail');

            return $media;
        });
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }
    public function taxLedgers()
    {
        return $this->belongsToMany(Ledger::class)->withPivot('rate');
    }


    public function taskEventLogs()
    {
        return $this->belongsToMany(TaskEventLog::class, 'product_task_event_logs');
    }

    public function productZonePivots()
    {
        return $this->hasMany(ProductZonePivot::class);
    }
    // belongsToMany Zone
    public function zones()
    {
        return $this->belongsToMany(Zone::class, 'product_zone_pivots')
            ->withPivot(['quantity', 'price']);
    }

    public function productZoneHistories()
    {
        return $this->hasMany(ProductZoneHistory::class);
    }

    // belongsToMany Circle
    public function circles()
    {
        return $this->belongsToMany(Circle::class, 'circle_product_pivots')
            ->withPivot(['quantity', 'price']);
    }
}
