<?php

namespace App\Models;

use App\Support\HasAdvancedFilter;
use DateTimeInterface;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;

class OInvoice extends Model
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;

    public const INV_TYPE_TASK = 'task';
    public const INV_TYPE_SERVICE = 'service';
    public const INV_TYPE_PRODUCT = 'product';

    public const INV_TYPES = [
        self::INV_TYPE_TASK => 'Normal',
        self::INV_TYPE_SERVICE => 'Service',
        // self::INV_TYPE_PRODUCT => 'Product',
    ];
    public $table = 'o_invoices';

    public $orderable = [
        'id',
        'slno',
        'invoice_no',
        'date',
    ];

    public $filterable = [
        'id',
        'slno',
        'invoice_no',
        'date',
        'clientGroup.name',
        'client.name',
    ];

    protected $fillable = [
        // 'fin_yr_id', 'slno', 'invoice_no', 'client_group_id', 'client_id', 'date', 'due_date',
        // 'client_name', 'client_address', 'client_email', 'client_phone', 'subtotal', 'taxes',
        // 'due', 'total', 'notes', 'extras', 'additional', 'tax_infos', 'status',

        'id', 'fin_yr_id', 'slno', 'invoice_no', 'client_group_id', 'client_id', 'date', 'due_date',
        'client_name', 'client_address', 'client_email', 'client_phone',
        'subtotal', 'taxes', 'due', 'total', 'notes', 'extras', 'additional', 'tax_infos', 'status',
        'invoiced_to_type', 'invoiced_to_id', 'invoiced_by_type', 'invoiced_by_id',
        'created_at', 'updated_at', 'deleted_at',

        'inv_type', 'party_name',
    ];

    protected $casts = [
        'date' => 'date',
        'due_date' => 'date',
        'extras' => 'array',
        'additional' => 'array',
        'tax_infos' => 'array',
    ];

    public function invoicedTo()
    {
        return $this->morphTo();
    }

    public function invoicedBy()
    {
        return $this->morphTo();
    }

    public function clientGroup()
    {
        return $this->belongsTo(ClientGroup::class, 'client_group_id');
    }

    public function client()
    {
        return $this->belongsTo(Client::class, 'client_id');
    }

    // public function invoiced_to()
    // {
    //     return $this->morphTo();
    // }

    // public function invoiced_by()
    // {
    //     return $this->morphTo();
    // }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    // date_f
    public function getDateFAttribute()
    {
        // return $this->attributes['date'] ? $this->attributes['date']->format('Y-m-d') : "";
        return  datePrint_dmY_hyphen($this->date);
    }

    // due_date_f
    public function getDueDateFAttribute()
    {
        return   datePrint_dmY_hyphen($this->due_date);
    }


    public function items()
    {
        return $this->hasMany(OInvoiceItem::class);
    }

    public function getSlnoFAttribute()
    {
        //get prefix from cache
        $prefix =  KVSetting::getInvoicePrefix();

        $slno = $this->attributes['slno'];

        // padd slno upto 8 digits
        // $slno = str_pad($slno, 6, '0', STR_PAD_LEFT);
        // $slno = str_pad($slno, 6, '0', STR_PAD_LEFT);

        return $prefix . $slno;
    }

    // set slno mutator
    public function setSlnoAttribute($value)
    {
        // $this->attributes['slno'] = $slno = KVSetting::getNextInvoiceSlno();
        $this->attributes['slno'] = $value;
        $this->attributes['invoice_no'] = $this->slno_f;
    }


    public static function CreateInvoiceM1($client_group_id, $client_id, $invoiceItems, $date = null, $due_date = null, $inv_type = self::INV_TYPE_TASK, $extraAttr = [])
    {
        // load clientGroup & client
        $clientGroup = ClientGroup::find($client_group_id);
        $client = Client::find($client_id);

        // get next slno
        $slno = KVSetting::getNextInvoiceSlno();
        $inv = new OInvoice();
        $inv->slno = $slno;
        $inv->client_group_id = $client_group_id;
        $inv->client_id = $client_id;
        $inv->client_name = Arr::get($extraAttr, 'party_name', $client->name);
        // $inv->client_name = $client->name;
        $inv->client_address = $client->rhq_address;
        $inv->client_email = $client->rhq_email;
        $inv->client_phone = $client->rhq_phone;
        $inv->date = $date;
        $inv->due_date = $due_date;
        $inv->inv_type = $inv_type;

        $extraFills = collect($extraAttr)->only([
            'party_name',
        ])->toArray();
        $inv->fill($extraFills);
        $inv->save();
        // laradump()->dump($invoiceItems);
        // add items
        foreach ($invoiceItems as $item) {
            $inv->items()->create($item);
        }

        // calculate totals
        $inv->calculateTotals();


        // find all invoiceItems where invoiceItem['type'] == 'task' and extract all invoiceItem['task_id']
        $taskIds = collect($invoiceItems)->where('type', 'task')->pluck('invoiceable_id')->all();
        // find all tasks where task['id'] in $taskIds && update all tasks with task['o_invoice_id'] = $inv->id
        // laradump()->dump(compact('taskIds', 'inv'));

        // dump(compact('taskIds', 'inv'));
        Task::whereIn('id', $taskIds)->update(['o_invoice_id' => $inv->id]);
    }

    public static function UpdateInvoiceM1(OInvoice $invoice, $client_group_id, $client_id, $invoiceItems, $date = null, $due_date = null, $inv_type = self::INV_TYPE_TASK, $extraAttr = [])
    {
        // load clientGroup & client
        $clientGroup = ClientGroup::find($client_group_id);
        $client = Client::find($client_id);

        // get next slno
        // $slno = KVSetting::getNextInvoiceSlno();
        $inv = $invoice;
        $inv->load(['items']);
        // $inv = new OInvoice();
        // $inv->slno = $slno;
        // $inv->client_group_id = $client_group_id;
        // $inv->client_id = $client_id;
        $inv->client_name = Arr::get($extraAttr, 'party_name', $client->name);
        $inv->client_address = $client->rhq_address;
        $inv->client_email = $client->rhq_email;
        $inv->client_phone = $client->rhq_phone;
        $inv->date = $date;
        $inv->due_date = $due_date;
        $inv->inv_type = $inv_type;

        $extraFills = collect($extraAttr)->only([
            'party_name',
        ])->toArray();
        $inv->fill($extraFills);
        $inv->update();

        // add items
        foreach ($invoiceItems as $item) {
            $itemId = Arr::get($item, "id");
            $hasId = $itemId != null;
            if ($hasId) {
                $invItem = $inv->items->find($itemId);
                $invItem->update($item);
            } else {
                $inv->items()->create($item);
            }
        }

        // calculate totals
        $inv->calculateTotals();


        $taskIds = collect($invoiceItems)->where('type', 'task')->pluck('invoiceable_id')->all();
        Task::whereIn('id', $taskIds)->update(['o_invoice_id' => $inv->id]);
    }

    public function calculateTotals()
    {
        $subtotal = 0;
        $taxes = 0;
        $total = 0;
        $due = 0;
        $totalGstAmount = 0;
        $totalCgstAmount = 0;
        $totalSgstAmount = 0;
        $totalIgstAmount = 0;
        foreach ($this->items as $item) {
            $subtotal       +=   floatVal($item->subtotal);
            $taxes          +=   floatVal($item->tax_amount);
            $total          +=   floatVal($item->total);
            $due            +=   floatVal($item->total);
            $totalGstAmount     +=  floatVal($item->gst_amount);
            $totalCgstAmount    +=  floatVal($item->cgst_amount);
            $totalSgstAmount    +=  floatVal($item->sgst_amount);
            $totalIgstAmount    +=  floatVal($item->igst_amount);
        }

        // get all unique tax rates for gst,sgst,cgst,igst
        $gst_rates = $this->items->pluck('gst_rate')->unique()->values()->all();
        $cgst_rates = $this->items->pluck('cgst_rate')->unique()->values()->all();
        $sgst_rates = $this->items->pluck('sgst_rate')->unique()->values()->all();
        $igst_rates = $this->items->pluck('igst_rate')->unique()->values()->all();




        // save tax_infos
        $tax_infos = [
            'total_gst_amount' => $totalGstAmount,
            'total_cgst_amount' => $totalCgstAmount,
            'total_sgst_amount' => $totalSgstAmount,
            'total_igst_amount' => $totalIgstAmount,
            'rates' => [
                'gst_rates' => $gst_rates,
                'cgst_rates' => $cgst_rates,
                'sgst_rates' => $sgst_rates,
                'igst_rates' => $igst_rates,
            ]
        ];
        $this->tax_infos = $tax_infos;

        $this->subtotal = $subtotal;
        $this->taxes = $taxes;
        $this->total = $total;

        $this->due = $due;
        $this->save();
    }

    public function getPdfUrlAttribute()
    {
        return route('admin.invoices.export', [
            'invoice' => $this->id
        ]);
    }


    // accessor is_service
    public function getIsServiceAttribute()
    {
        return $this->inv_type ==  self::INV_TYPE_SERVICE;
    }

    // accessor is_task
    public function getIsTaskAttribute()
    {
        return empty($this->inv_type) || $this->inv_type == self::INV_TYPE_TASK;
    }

    // accessor is_product
    public function getIsProductAttribute()
    {
        return $this->inv_type == self::INV_TYPE_PRODUCT;
    }
}
