<?php

namespace App\Models;

use \DateTimeInterface;
use App\Support\HasAdvancedFilter;
use App\Traits\Auditable;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class EmployeeAttendanceLog extends Model implements HasMedia
{
    use HasFactory;
    use HasAdvancedFilter;
    use SoftDeletes;
    use InteractsWithMedia;
    use Auditable;


    public const ATTENDANCE_FROM_SELECT = [
        'staff_app' => 'Staff App',
        'system'    => 'System',
        'manual'    => 'Manual',
    ];


    public const TYPE_RADIO = [
        'present'      => 'Present',
        'absent'       => 'Absent',
        // 'half-day'     => 'Half Day',
        'late'         => 'Late',
        'holiday'      => 'Holiday',
        'paid-leave'   => 'Paid Leave',
        'unpaid-leave' => 'Unpaid Leave',
    ];

    public $table = 'employee_attendance_logs';

    public $orderable = [
        'id',
        'date',
        'user.name',
        'user.email',
        'type',
        'coord_lat',
        'coord_long',
        'coord_accu',
        'location',
        'signed_in_at',
        'signed_out_at',
        'device',
        'attendance_from',
    ];

    public $filterable = [
        'id',
        'date',
        'user.name',
        'user.email',
        'type',
        'coord_lat',
        'coord_long',
        'coord_accu',
        'location',
        'signed_in_at',
        'signed_out_at',
        'device',
        'attendance_from',
        'task.name',
    ];

    protected $fillable = [
        'date',
        'user_id',
        'type',
        'coord_lat',
        'coord_long',
        'coord_accu',
        'location',
        'coord_data',
        'geocode_data',
        'signed_in_at',
        'signed_out_at',
        'device',
        'attendance_from',
    ];

    protected $appends = [
        'sign_in_photo',
        'sign_out_photo',
    ];
    protected $dates = [
        'date',
        'signed_in_at',
        'signed_out_at',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'coord_data' => 'array',
        'geocode_data' => 'array',
    ];

    public function registerMediaConversions(Media $media = null): void
    {
        $thumbnailWidth  = 50;
        $thumbnailHeight = 50;

        $thumbnailPreviewWidth  = 120;
        $thumbnailPreviewHeight = 120;

        $this->addMediaConversion('thumbnail')
            ->width($thumbnailWidth)
            ->height($thumbnailHeight)
            ->fit('crop', $thumbnailWidth, $thumbnailHeight);
        $this->addMediaConversion('preview_thumbnail')
            ->width($thumbnailPreviewWidth)
            ->height($thumbnailPreviewHeight)
            ->fit('crop', $thumbnailPreviewWidth, $thumbnailPreviewHeight);
    }

    public function getDateAttribute($value)
    {
        return $value ? Carbon::parse($value)->format(config('project.date_format')) : null;
    }

    public function setDateAttribute($value)
    {
        $this->attributes['date'] = $value ? Carbon::createFromFormat(config('project.date_format'), $value)->format('Y-m-d') : null;
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function getTypeLabelAttribute($value)
    {
        return static::TYPE_RADIO[$this->type] ?? null;
    }

    public function getSignedInAtAttribute()
    // public function getSignedInAtAttribute($value)
    {
        $value = /* $this->getAttribute('signed_in_at') ?? */ $this->getAttribute('created_at');
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setSignedInAtAttribute($value)
    {
        $this->attributes['signed_in_at'] = $value ? Carbon::createFromFormat(config('project.datetime_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    public function getSignedOutAtAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('project.datetime_format')) : null;
    }

    public function setSignedOutAtAttribute($value)
    {
        $this->attributes['signed_out_at'] = $value ? Carbon::createFromFormat(config('project.datetime_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    public function getSignInPhotoAttribute()
    {
        return $this->getMedia('employee_attendance_log_sign_in_photo')->map(function ($item) {
            $media = $item->toArray();
            $media['url'] = $item->getUrl();
            $media['thumbnail'] = $item->getUrl('thumbnail');
            $media['preview_thumbnail'] = $item->getUrl('preview_thumbnail');

            return $media;
        });
    }

    public function getSignOutPhotoAttribute()
    {
        return $this->getMedia('employee_attendance_log_sign_out_photo')->map(function ($item) {
            $media = $item->toArray();
            $media['url'] = $item->getUrl();
            $media['thumbnail'] = $item->getUrl('thumbnail');
            $media['preview_thumbnail'] = $item->getUrl('preview_thumbnail');

            return $media;
        });
    }

    public function getAttendanceFromLabelAttribute($value)
    {
        return static::ATTENDANCE_FROM_SELECT[$this->attendance_from] ?? null;
    }

    public function task()
    {
        return $this->belongsToMany(Task::class);
    }

    // hasOne
    public function reimbursement()
    {
        return $this->hasOne(VisitReimbursement::class, 'visit_id', 'id');
    }

    protected function serializeDate(DateTimeInterface $date)
    {
        return $date->format('Y-m-d H:i:s');
    }

    #accessor gmap_url1
    public function getGmapUrl1Attribute()
    {
        // https://www.google.com/maps/@23.1310146,88.6320453,15z
        if ($this->coord_lat && $this->coord_long) {
            return 'https://www.google.com/maps/@' . $this->coord_lat . ',' . $this->coord_long . ',15z';
        }
    }
    #accessor gmap_url2
    public function getGmapUrl2Attribute()
    {
        // https://www.google.com/maps/place/Gangnapur,+West+Bengal+741238/@23.1310146,88.6320453,15z/
        if ($this->coord_lat && $this->coord_long && $this->location) {
            return 'https://www.google.com/maps/place/' . urlencode($this->place) . '/' . $this->coord_lat . ',' . $this->coord_long . ',15z/';
        }
    }

    #accessor place
    public function getPlaceAttribute()
    {
        $fi = Arr::get($this->geocode_data, '0');
        if ($fi) {
            return join(", ", [
                Arr::get($fi, 'subThoroughfare'),
                Arr::get($fi, 'thoroughfare'),
                Arr::get($fi, 'subLocality'),
            ]);
        }
    }
}
