<?php

namespace App\Http\Livewire\EmployeeAttendanceLog;

use App\Models\EmployeeAttendanceLog;
use App\Models\Task;
use App\Models\User;
use Livewire\Component;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class Edit extends Component
{
    public array $task = [];

    public array $mediaToRemove = [];

    public array $listsForFields = [];

    public array $mediaCollections = [];

    public EmployeeAttendanceLog $employeeAttendanceLog;

    public function addMedia($media): void
    {
        $this->mediaCollections[$media['collection_name']][] = $media;
    }

    public function removeMedia($media): void
    {
        $collection = collect($this->mediaCollections[$media['collection_name']]);

        $this->mediaCollections[$media['collection_name']] = $collection->reject(fn ($item) => $item['uuid'] === $media['uuid'])->toArray();

        $this->mediaToRemove[] = $media['uuid'];
    }

    public function getMediaCollection($name)
    {
        return $this->mediaCollections[$name];
    }

    public function mount(EmployeeAttendanceLog $employeeAttendanceLog)
    {
        $this->employeeAttendanceLog = $employeeAttendanceLog;
        $this->task                  = $this->employeeAttendanceLog->task()->pluck('id')->toArray();
        $this->initListsForFields();
        $this->mediaCollections = [
            'employee_attendance_log_sign_in_photo'  => $employeeAttendanceLog->sign_in_photo,
            'employee_attendance_log_sign_out_photo' => $employeeAttendanceLog->sign_out_photo,
        ];
    }

    public function render()
    {
        return view('livewire.employee-attendance-log.edit');
    }

    public function submit()
    {
        $this->validate();

        $this->employeeAttendanceLog->save();
        $this->employeeAttendanceLog->task()->sync($this->task);
        $this->syncMedia();

        return redirect()->route('admin.employee-attendance-logs.index');
    }

    protected function syncMedia(): void
    {
        collect($this->mediaCollections)->flatten(1)
            ->each(fn ($item) => Media::where('uuid', $item['uuid'])
            ->update(['model_id' => $this->employeeAttendanceLog->id]));

        Media::whereIn('uuid', $this->mediaToRemove)->delete();
    }

    protected function rules(): array
    {
        return [
            'employeeAttendanceLog.date' => [
                'required',
                'date_format:' . config('project.date_format'),
            ],
            'employeeAttendanceLog.user_id' => [
                'integer',
                'exists:users,id',
                'nullable',
            ],
            'employeeAttendanceLog.type' => [
                'nullable',
                'in:' . implode(',', array_keys($this->listsForFields['type'])),
            ],
            'employeeAttendanceLog.coord_lat' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.coord_long' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.coord_accu' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.location' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.signed_in_at' => [
                'nullable',
                'date_format:' . config('project.datetime_format'),
            ],
            'employeeAttendanceLog.signed_out_at' => [
                'nullable',
                'date_format:' . config('project.datetime_format'),
            ],
            'mediaCollections.employee_attendance_log_sign_in_photo' => [
                'array',
                'nullable',
            ],
            'mediaCollections.employee_attendance_log_sign_in_photo.*.id' => [
                'integer',
                'exists:media,id',
            ],
            'mediaCollections.employee_attendance_log_sign_out_photo' => [
                'array',
                'nullable',
            ],
            'mediaCollections.employee_attendance_log_sign_out_photo.*.id' => [
                'integer',
                'exists:media,id',
            ],
            'employeeAttendanceLog.device' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.attendance_from' => [
                'nullable',
                'in:' . implode(',', array_keys($this->listsForFields['attendance_from'])),
            ],
            'task' => [
                'array',
            ],
            'task.*.id' => [
                'integer',
                'exists:tasks,id',
            ],
        ];
    }

    protected function initListsForFields(): void
    {
        $this->listsForFields['user']            = User::pluck('name', 'id')->toArray();
        $this->listsForFields['type']            = $this->employeeAttendanceLog::TYPE_RADIO;
        $this->listsForFields['attendance_from'] = $this->employeeAttendanceLog::ATTENDANCE_FROM_SELECT;
        $this->listsForFields['task']            = Task::pluck('name', 'id')->toArray();
    }
}
