<?php

namespace App\Http\Livewire\EmployeeAttendanceLog;

use App\Models\EmployeeAttendanceLog;
use App\Models\Task;
use App\Models\User;
use Carbon\Carbon;
use Livewire\Component;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class Create extends Component
{
    public array $task = [];

    public array $mediaToRemove = [];

    public array $listsForFields = [];

    public array $mediaCollections = [];

    public EmployeeAttendanceLog $employeeAttendanceLog;

    public function addMedia($media): void
    {
        $this->mediaCollections[$media['collection_name']][] = $media;
    }

    public function removeMedia($media): void
    {
        $collection = collect($this->mediaCollections[$media['collection_name']]);

        $this->mediaCollections[$media['collection_name']] = $collection->reject(fn ($item) => $item['uuid'] === $media['uuid'])->toArray();

        $this->mediaToRemove[] = $media['uuid'];
    }

    public function mount(EmployeeAttendanceLog $employeeAttendanceLog)
    {
        $this->employeeAttendanceLog       = $employeeAttendanceLog;
        $this->employeeAttendanceLog->type = 'present';
        $this->employeeAttendanceLog->attendance_from = 'manual';
        $this->initListsForFields();
    }

    public function render()
    {
        return view('livewire.employee-attendance-log.create');
    }

    public function submit()
    {
        $this->validate();
        if ($this->validateSameDayAttendanceForAUser()) {
            $this->employeeAttendanceLog->save();
            $this->employeeAttendanceLog->task()->sync($this->task);
            $this->syncMedia();

            return redirect()->route('admin.employee-attendance-logs.index');
        }

    }

    public function validateSameDayAttendanceForAUser()
    {
        // dump($this->employeeAttendanceLog->date);
        $date = Carbon::createFromFormat(config('project.date_format'), $this->employeeAttendanceLog->date);
        $count = EmployeeAttendanceLog::/* whereNull('task_id')-> */where('user_id', $this->employeeAttendanceLog->user_id)
            ->whereBetween('date', [(clone $date)->startOfDay(), (clone $date)->endOfDay()])->count();

        if ($count > 0) {
            $this->addError('employeeAttendanceLog.user_id', 'Already signed in for same day.');
            return false;
        }
        return true;
    }

    protected function syncMedia(): void
    {
        collect($this->mediaCollections)->flatten(1)
            ->each(fn ($item) => Media::where('uuid', $item['uuid'])
                ->update(['model_id' => $this->employeeAttendanceLog->id]));

        Media::whereIn('uuid', $this->mediaToRemove)->delete();
    }

    protected function rules(): array
    {
        return [
            'employeeAttendanceLog.date' => [
                'required',
                'date_format:' . config('project.date_format'),
            ],
            'employeeAttendanceLog.user_id' => [
                'integer',
                'exists:users,id',
                'nullable',
            ],
            'employeeAttendanceLog.type' => [
                'nullable',
                'in:' . implode(',', array_keys($this->listsForFields['type'])),
            ],
            'employeeAttendanceLog.coord_lat' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.coord_long' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.coord_accu' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.location' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.signed_in_at' => [
                'nullable',
                'date_format:' . config('project.datetime_format'),
            ],
            'employeeAttendanceLog.signed_out_at' => [
                'nullable',
                'date_format:' . config('project.datetime_format'),
            ],
            'mediaCollections.employee_attendance_log_sign_in_photo' => [
                'array',
                'nullable',
            ],
            'mediaCollections.employee_attendance_log_sign_in_photo.*.id' => [
                'integer',
                'exists:media,id',
            ],
            'mediaCollections.employee_attendance_log_sign_out_photo' => [
                'array',
                'nullable',
            ],
            'mediaCollections.employee_attendance_log_sign_out_photo.*.id' => [
                'integer',
                'exists:media,id',
            ],
            'employeeAttendanceLog.device' => [
                'string',
                'nullable',
            ],
            'employeeAttendanceLog.attendance_from' => [
                'nullable',
                'in:' . implode(',', array_keys($this->listsForFields['attendance_from'])),
            ],
            'task' => [
                'array',
            ],
            'task.*.id' => [
                'integer',
                'exists:tasks,id',
            ],
        ];
    }

    protected function initListsForFields(): void
    {
        $this->listsForFields['user']            = User::whereHas('roles', function ($query) {
            $query->whereIn('role_id', [7, 5, 4, 3, 2, 1]);
        })
            ->pluck('name', 'id')
            ->toArray();
        $this->listsForFields['type']            = $this->employeeAttendanceLog::TYPE_RADIO;
        $this->listsForFields['attendance_from'] = $this->employeeAttendanceLog::ATTENDANCE_FROM_SELECT;
        $this->listsForFields['task']            = Task::pluck('name', 'id')->toArray();
    }
}
